/* -------------------------------------------------------------------
  Titre: Serveur HTTP V6 (dossier GENERIQUE)
  Date : 23/4/2025 - mise à jour des bibliothèques
  Objectifs:
  Commander la led L et les sorties D0, D1, D2, D3 d'une carte MKR Wifi 1010,
  et obtenir les valeurs présentes sur les entrées A0,..., A4 à 
  partir d'un navigateur (requêtes GET ou POST).
  URL reconnues:
  - http://IP demande "La page d'accueil"
  - http://IP/lire/val0 demande la valeur sur A0
  - http://IP/lire/val1 demande la valeur sur A1
  - http://@IP/lire/vals0et1 demande les valeurs sur les entrées A0 et A1 (json)
  - http://IP/lire/val2 demande la valeur sur A2
  - http://IP/lire/val3 demande la valeur sur A3
  - http://IP/lire/val4 demande la valeur sur A4
  - http://IP/lire/all demande toutes les valeurs sur les entrées A0 à An (json)
  - http://IP/lire/tablevar demande une table, 24 lignes (0 à 23) x 2 colonnes, de valeurs numériques (0<-A0,1<-A1) et 22constantes (max=50)
  - http://IP/ecrire/valA active ou désactive la LED L et D0
  - http://IP/ecrire/valB active ou désactive D1
  - http://IP/ecrire/valC active ou désactive D2
  - http://IP/ecrire/valD active ou désactive D3
  Bibliothèques :
  - WiFiWebServer, WIFININA, SPI : appelées dans defines.h
  - Documentation de WiFiWebServer et WifiNINA
    - https://github.com/khoih-prog/WiFiWebServer
    - https://www.arduino.cc/en/Reference/WiFiNINA
  Carte arduino:
  - Module NINA (Arduino MKR WiFi 1010, MKR VIDOR 4000 et UNO WiFi Rev.2)
  Fichiers
  - define.h, Server_HTTP_V6
  -----------------------------------------------------------------------*/
// Définitions
#define LED LED_BUILTIN  //  LED_BUILTIN (L sur la carte !)
#define D0 0
#define D1 1
#define D2 2
#define D3 3

// Bibliothèques à ajouter
#include "defines.h"

// Page d'accueil
const String paccueil =
  F("<html>\
<head><meta charset=\"utf-8\">\
<title>Serveur MKR Wifi 1010 V6</title>\
<style>\
.ip{color:blue;}\
table {text-align: center;}\
td {column-width: 50px;}\
a {text-decoration: none;}\
</style>\
</head>\
<body>\
<h3>[Accueil] Serveur HTTP <span style=\"color:red;\">V6</span> (MKR Wifi 1010)</h3>\
<b>Commandes reconnues</b>\
<ul>\
<li><em><b>Ecriture</b> sur les sorties numériques <b>D0</b>(LED <b>L</b>) <- valA  | <b>D1</b> <- valB | <b>D2</b> <- valC | <b>D3</b> <- valD<br>\
 => http://IP/ecrire/valx?val=0 ou 1 avec x=A,B,C ou D</em></li><br>\
    <table>\
        <tr>\
            <td>DO</td>\
            <td>D1</td>\
            <td>D2</td>\
            <td>D3</td>\
        </tr>\
        <tr>\
            <td><button><a href=\"/ecrire/valA?val=1\">On</a></button></td>\
            <td><button><a href=\"/ecrire/valB?val=1\">On</a></button></td>\
            <td><button><a href=\"/ecrire/valC?val=1\">On</a></button></td>\
            <td><button><a href=\"/ecrire/valD?val=1\">On</a></button></td>\
        </tr>\
        <tr>\
            <td><button><a href=\"/ecrire/valA?val=0\">Off</a></button></td>\
            <td><button><a href=\"/ecrire/valB?val=0\">Off</a></button></td>\
            <td><button><a href=\"/ecrire/valC?val=0\">Off</a></button></td>\
            <td><button><a href=\"/ecrire/valD?val=0\">Off</a></button></td>\
        </tr>\
    </table><br>\
<li><em><b>Lecture</b> des entrées analogiques : <b>A0</b> -> val0, ... , <b>A4</b> -> val4</em>\
 <ul>\
    <li><b>Unique</b>\
      <ul>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/val0\">/lire/val0</a></span></li>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/val1\">/lire/val1</a></span></li>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/val2\">/lire/val2</a></span></li>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/val3\">/lire/val3</a></span></li>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/val4\">/lire/val4</a></span></li>\
      </ul>\
    </li><br>\
    <li><b>Multiples JSON</b>\
      <ul>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/vals0et1\">/lire/vals0et1</a></span></li>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/all\">/lire/all</a></span></li>\
        <li>http://IP<span class=\"ip\"><a href=\"/lire/table\">/lire/tablevar</a></span>\
        <b>Table</b>(24 lignes * 2 colonnes) ligne0<-A0, ligne1<-A1 et 22 constantes (valeur max=50)</li>\
      </ul>\
    </li>\
  </ul></li></ul>\
</body></html>");

// Paramètres réseau, cryptage WPA
const char ssid[] = "SynBoxLAN";  // nom du SSID
const char pass[] = "?????";  // mot de passe du réseau

// Création du serveur HTTP
int status = WL_IDLE_STATUS;
WiFiWebServer server(80);

// Grandeurs physiques à mesurer (simulées par potentiomètres)
unsigned int val0 = 0;
unsigned int val1 = 0;
unsigned int val2 = 0;
unsigned int val3 = 0;
unsigned int val4 = 0;

// Gestionnaires de requête
//----------------------------------------------------------------
// g0. Page d'accueil
void handleRoot() {
  server.send(200, F("text/html"), paccueil);  // Réponse
}

// g1. Commande de la led L de la carte et de la sortie D0
void ecritureValA() {
  if (server.arg(0) == "0") {  // Test de la valeur du premier argument
    digitalWrite(LED, LOW);    // Désactivation de la commande de la LED L
    digitalWrite(D0, LOW);     // et de la sortie D0
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else if (server.arg(0) == "1") {
    digitalWrite(LED, HIGH);  // Activation de la commande de la LED L
    digitalWrite(D0, HIGH);   // et de la sortie D0
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else {
    server.send(200, F("text/plain"), String("Commande <-  0 ou 1 !!!"));
  }
}

// g2. Commande de la sortie D1
void ecritureValB() {
  if (server.arg(0) == "0") {  // Test de la valeur du premier argument
    digitalWrite(D1, LOW);     // Désactivation de la sortie D1
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else if (server.arg(0) == "1") {
    digitalWrite(D1, HIGH);  // Activation de la sortie D1
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else {
    server.send(200, F("text/plain"), String("Commande <-  0 ou 1 !!!"));
  }
}

// g3. Commande de la sortie D2
void ecritureValC() {
    if (server.arg(0) == "0") {  // Test de la valeur du premier argument
    digitalWrite(D2, LOW);     // Désactivation de la sortie D1
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else if (server.arg(0) == "1") {
    digitalWrite(D2, HIGH);  // Activation de la sortie D1
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else {
    server.send(200, F("text/plain"), String("Commande <-  0 ou 1 !!!"));
  }
}

// g4. Commande de la sortie D3
void ecritureValD() {
      if (server.arg(0) == "0") {  // Test de la valeur du premier argument
    digitalWrite(D3, LOW);     // Désactivation de la sortie D1
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else if (server.arg(0) == "1") {
    digitalWrite(D3, HIGH);  // Activation de la sortie D1
    server.send(200, F("text/plain"), String(server.arg(0)));
  } else {
    server.send(200, F("text/plain"), String("Commande <-  0 ou 1 !!!"));
  }
}

// g5. Mesure et transmission de la valeur analogique présente sur A0
void lectureVal0() {
  unsigned int val = val0;
  server.send(200, F("text/plain"), String(val));
}

// g6. Mesure et transmission de la valeur analogique présente sur A1
void lectureVal1() {
  unsigned int val = val1;
  server.send(200, F("text/plain"), String(val));
}

// g7. Mesure et transmission de la valeur analogique présente sur A2
void lectureVal2() {
  unsigned int val = val2;
  server.send(200, F("text/plain"), String(val));
}

// g8. Mesure et transmission de la valeur analogique présente sur A3
void lectureVal3() {
  unsigned int val = val3;
  server.send(200, F("text/plain"), String(val));
}

// g9. Mesure et transmission de la valeur analogique présente sur A4
void lectureVal4() {
  unsigned int val = val4;
  server.send(200, F("text/plain"), String(val));
}

// g10. Mesure et transmission des valeurs analogiques présentes sur A0 et A1 au format JSON
void lectureVals0et1()
{
  unsigned int val_0 = val0;
  unsigned int val_1 = val1;
  String msgjson = "[{\"val0\":";
  msgjson += val_0;
  msgjson += ",";
  msgjson += "\"val1\":";
  msgjson += val_1;
  msgjson += "}]";
  server.send(200, F("text/plain"), msgjson);
}

// g11. Mesure et transmission des valeurs analogiques présentes sur A0 à An au format JSON
void lectureAll() {
  // Mesures
  unsigned int val_0 = val0;
  unsigned int val_1 = val1;
  unsigned int val_2 = val2;
  unsigned int val_3 = val3;
  unsigned int val_4 = val4;
  String msgjson = "[{\"val0\":";
  msgjson += val_0;
  msgjson += ",";
  msgjson += "\"val1\":";
  msgjson += val_1;
  msgjson += ",";
  msgjson += "\"val2\":";
  msgjson += val_2;
  msgjson += ",";
  msgjson += "\"val3\":";
  msgjson += val_3;
  msgjson += ",";
  msgjson += "\"val4\":";
  msgjson += val_4;
  msgjson += "}]";
  server.send(200, F("text/plain"), msgjson);
}


// g12. Mesure de 2 valeurs mises à l'echelle (max 50)
// et insertion dans une table JSON
void lectureTableJSON() {
  unsigned int val_0 = val0;
  unsigned int val_1 = val1;
  String msgtable = R"([{ "H0": 0, "T0": )" + String(val_0 * 0.05) + R"( }, { "H1": 1, "T1": )" + String(val_1 * 0.05) +
  R"( }, { "H": 1, "T": 11.7 }, 
  { "H": 2, "T": 12 }, 
  { "H": 3, "T": 12 }, 
  { "H": 4, "T": 12 }, 
  { "H": 5, "T": 10.7 }, 
  { "H": 6, "T": 10.6 }, 
  { "H": 7, "T": 10.1 }, 
  { "H": 8, "T": 9.7 }, 
  { "H": 9, "T": 10.5 }, 
  { "H": 10, "T": 11.4 }, 
  { "H": 11, "T": 13.1 }, 
  { "H": 12, "T": 15.7 }, 
  { "H": 13, "T": 19.2 }, 
  { "H": 14, "T": 20.1 }, 
  { "H": 15, "T": 20.9 }, 
  { "H": 16, "T": 20.7 }, 
  { "H": 17, "T": 20.9 }, 
  { "H": 18, "T": 20.7 }, 
  { "H": 19, "T": 19.1 }, 
  { "H": 20, "T": 17.4 }, 
  { "H": 21, "T": 16.5 }, 
  { "H": 22, "T": 16.6 }, 
  { "H": 23, "T": 15.4 }])";

  // Transmission
  server.send(200, F("text/plain"), msgtable);
}

// gn. Traitement des requêtes non prises en charge
void handleNotFound() {
  String message = F("File Not Found\n\n");
  message += F("URI: ");
  message += server.uri();
  message += F("\nMethod: ");
  message += (server.method() == HTTP_GET) ? F("GET") : F("POST");
  message += F("\nArguments: ");
  message += server.args();
  message += F("\n");
  for (uint8_t i = 0; i < server.args(); i++) {
    message += " " + server.argName(i) + ": " + server.arg(i) + "\n";
  }
  server.send(404, F("text/plain"), message);
}
//----------------------------------------------------------------

void setup() {
  // Configuration des E/S
  pinMode(LED, OUTPUT);  // Led L de la carte
  pinMode(D0, OUTPUT);
  pinMode(D1, OUTPUT);

  Serial.begin(115200);  // Moniteur pour la mise au point

  // 1. Vérification de la connexion au module Wifi
  if (WiFi.status() == WL_NO_MODULE) {
    Serial.println(" La communication avec le module WiFi a échoué!");
    // ATTENTION : blocage du programme !!!!
    while (true)
      ;
  }

  // 2. Connexion au réseau Wifi
  while (status != WL_CONNECTED) {
    Serial.print("Tentative de connexion au réseau: ");
    Serial.println(ssid);
    status = WiFi.begin(ssid, pass);  // si WPA, WPA2
    // Attente de 10s avant la prochaine tentative de connexion
    delay(10000);
  }

  // 3. Connecté, inscription des gestionnaires
  server.on(F("/"), handleRoot);
  server.on(F("/lire/val0"), lectureVal0);
  server.on(F("/lire/val1"), lectureVal1);
  server.on(F("/lire/val2"), lectureVal2);
  server.on(F("/lire/val3"), lectureVal3);
  server.on(F("/lire/val4"), lectureVal4);
  server.on(F("/lire/vals0et1"), lectureVals0et1);
  server.on(F("/lire/all"), lectureAll);
  server.on(F("/lire/table"), lectureTableJSON),
  server.on(F("/ecrire/valA"), ecritureValA);
  server.on(F("/ecrire/valB"), ecritureValB);
  server.on(F("/ecrire/valC"), ecritureValC);
  server.on(F("/ecrire/valD"), ecritureValD);

  server.onNotFound(handleNotFound);

  // et démarrage du serveur Web sur le port défini
  // lors de l'initialisation
  server.begin();
}

// Paramètres de la tâche non bloquante
unsigned long previousMillis = 0;  // Temps précédent
const long interval = 1000;        // Intervalle en millisecondes (1 seconde)

void loop() {
  server.handleClient();
  // TODO mesures avec millis() et remplissage table
  unsigned long currentMillis = millis();

  // Tâche non bloquante
  if (currentMillis - previousMillis >= interval) {
    previousMillis = currentMillis;
    // Mesures
    val0 = analogRead(A0);
    val1 = analogRead(A1);
    val2 = analogRead(A2);
    val3 = analogRead(A3);
    val4 = analogRead(A4);
    // Indicateur fonctionnement du programme
    digitalWrite(LED, !digitalRead(LED));
  }
}
