/* ---------------------------------------------------------------
Création le 10/8/2019
Mise à jour le 25/2/2021
Source : Hackable Magazine n°26
Auteur : Denis Bodor
Titre : Client MQTT
Fonction : template
Description : Publication d'une valeur incrémentée et abonnement
à une commande on/off destinée à commander la LED de la carte
Editeur : rédigé avec VSCode
Cible : ESP8266
Fichiers : ARD_ESP8266_MQTT_REF.ino, parametres.h
----------------------------------------------------------------*/
// ---------------------------------------------------------------------------------
// Bibliothèques
// ---------------------------------------------------------------------------------
// MQTT : implémentation d'un client MQTT
#include <PubSubClient.h>
// Connexion au wifi
#include <ESP8266WiFi.h>
// mDNS pour la résolution des noms des hôtes
#include <ESP8266mDNS.h>
//  EEPROM : émule une EEPROM dans l'ESP8266
#include <EEPROM.h>
// Paramètres de connexion au broker
#include "parametres.h"
// ---------------------------------------------------------------------------------
// Paramètres pour la connexion au Broker A RENSEIGNER dans parametres.h
// NE PAS MODIFIER le code ci-dessous
// ---------------------------------------------------------------------------------
const char *mqtt_server = BROKER;
uint16_t mqtt_PORT = MQTTPORT; // Port TCP sur lequel le broker écoute
// (par défaut pour brokers MQTT)
const uint8_t qos = QoS; // Qualité de service
// ---------------------------------------------------------------------------------
// Les topics (in = Abonnement / out = publication) SONT A DEFINIR dans parametres.h
// #### Constantes ci-dessous A ADAPTER en fonction du fichier parametres.h ####
// ---------------------------------------------------------------------------------
const char *inTopic = TOPIC_IN;
const char *outTopic = TOPIC_OUT;
// ---------------------------------------------------------------------------------
// Structure pour la configuration de la connexion au réseau wifi
struct EEconf
{ // Les champs sont remplis par le croquis infoClientMQTT_ESP8266.ino
    // avec les données stockées dans l'EEPROM (émulée)
    char ssid[32];       // SSID du réseau. Exemple : SynBoxLAN,
    char password[64];   // Mot de passe du réseau. Exemple : 12345678
    char myhostname[32]; // Nom donné au client MQTT. Exemple : ESP8266_1
} readconf;
// Objet pour la connexion au réseau wifi
WiFiClient espClient;
// Objet pour la connexion au broker MQTT (Publisher/Subscriber)
PubSubClient mqttClient(espClient);
// Intervalle de temps séparant la publication du (des) topic()s
const long interval = INTERVAL; // A RENSEIGNER dans parametres.h
// Permet de calculer l'intervalle de temps depuis la dernière
// publication du (des) topic(s)
unsigned long previousMillis = 0;
// Valeur à publier (pour tester la connexion)
uint32_t compteur = 0;
// ---------------------------------------------------------------------------------
// Fonctions
// ---------------------------------------------------------------------------------
// Connexion au Wifi
// ---------------------------------------------------------------------------------
void setup_wifi()
{
    // Mode station
    WiFi.mode(WIFI_STA);
    Serial.println();
    Serial.print("Tentative de connexion à ");
    Serial.println(readconf.ssid);
    // Connexion au Wifi
    WiFi.begin(readconf.ssid, readconf.password);
    while (WiFi.status() != WL_CONNECTED)
    {
        delay(5000);
        Serial.print(".");
    }
    // Affichage
    Serial.println("");
    Serial.println("Connexion au Wifi ok");
    Serial.print("Adresse IP : ");
    Serial.println(WiFi.localIP());
    // Configuration de mDNS
    WiFi.hostname(readconf.myhostname);
    if (!MDNS.begin(readconf.myhostname))
    {
        Serial.println("Erreur de configuration mDNS !");
    }
    else
    {
        Serial.println("Répondeur mDNS démarré");
        Serial.println(readconf.myhostname);
    }
}
// ---------------------------------------------------------------------------------
// Traitement des messages provenant du broker
// #### A ADAPTER en fonction du ou des messages reçus ####
// Dans cette exemple, on commande la led BUILTIN de la carte
// ---------------------------------------------------------------------------------
// Fonction de rappel (appelée automatiquement lors de la réception d'un message)
// topic : chaîne de caractères décrivant le topic
// payload : message associé au topic
// length : nombre d'octets du message
void callback(char *topic, byte *payload, unsigned int length)
{
    Serial.print("Message [");
    Serial.print(topic);
    Serial.print("] ");
    // Affichage du message
    for (int i = 0; i < length; i++)
    {
        Serial.print((char)payload[i]);
    }
    Serial.println("");
    // '1' est-il le premier caractère du message ?
    if ((char)payload[0] == '1')
    {
        // Oui led=on
        digitalWrite(BUILTIN_LED, LOW);
    }
    else
    {
        // Non led=off
        digitalWrite(BUILTIN_LED, HIGH);
    }
}
// ---------------------------------------------------------------------------------
// Maintien de la connexion au broker MQTT et abonnement au(x) topic(s)
// #### A ADAPTER en fonction du nombre d'abonnements  ####
// ---------------------------------------------------------------------------------
void reconnect()
{
    // Connecté au broker ?
    while (!mqttClient.connected())
    {
        // Non, on se connecte
        if (!mqttClient.connect(readconf.myhostname))
        {
            Serial.print("Erreur de connexion MQTT, rc=");
            Serial.println(mqttClient.state());
            delay(5000);
            continue;
        }
        Serial.println("Connexion serveur MQTT ok");
        // Connecté ! On s'abonne au topic (ici "ctrl_led")
        mqttClient.subscribe(inTopic, qos);
        Serial.print("Abonnement au topic ");
        Serial.println(inTopic);
    }
}
// ---------------------------------------------------------------------------------
// Programme principal : initialisation
// ---------------------------------------------------------------------------------
void setup()
{
    // Configuration de la broche connectée à la LED de la carte
    pinMode(BUILTIN_LED, OUTPUT);
    // Configuration du moniteur série
    Serial.begin(115200);
    while (!Serial)
        ; // Blocage
    // Lecture des paramètres sauvegardés par ARD_ESP_SauveInfosClientMqtt.ino
    EEPROM.begin(sizeof(readconf));
    EEPROM.get(0, readconf);
    // Connexion au Wifi
    setup_wifi();
    // Configuration du broker
    // MQTT_PORT = 1883 (port TCP par défaut des brokers MQTT)
    mqttClient.setServer(mqtt_server, mqtt_PORT);
    // Activation de la fonction callback
    mqttClient.setCallback(callback);
}
// ---------------------------------------------------------------------------------
// Programme principal : boucle infinie
// ---------------------------------------------------------------------------------
void loop()
{
    // Tableau(x) pour la mise en forme de la (ou des) valeur(s) publiée(s)
    char payload[16];
    // Tableau(x) pour le(s) topic(s) construit(s) avec le nom de l'hôte
    char topic[64];

    // Sommes-nous connectés au broker MQTT ?
    if (!mqttClient.connected())
    {
        // Non alors
        reconnect(); // maintenir la connexion et s'abonner à un ou plusieurs topics
    }
    // Oui, maintien de la connexion avec le broker
    // Interrogation du broker : ne doit pas être bloquée !!! pour traiter les messages reçus
    mqttClient.loop(); // Déclencle la fonction callback si un message est reçu

    // Temporisation "non blocante" avant la publication d'un ou plusieurs message
    long currentMillis = millis();
    if (currentMillis - previousMillis > interval) // si temporisation > interval en ms
    {                                              // alors on publie le(s) message(s)
        // On mémorise la "date" à laquelle le ou les messages ont été publiés
        previousMillis = currentMillis;
        compteur++; // Incrémentation de la valeur à publier (test)
        // Construction du ou des message(s)
        sprintf(payload, "%hu", compteur);
        // Construction du (des) topic(s) avec le nom d'hôte 
        // Exemple "<myhostname>/<valeur>" #### A ADAPTER ####
        sprintf(topic, "%s/%s", readconf.myhostname, outTopic);
        // Publication du (des) message(s) sur le(s) topic(s)
        mqttClient.publish(topic, payload);
    }
}
// ---------------------------------------------------------------------------------
