/**
 * Script de validation du formulaire de connexion
 * Vérifie les champs avant l'envoi des données au serveur
 */

// Attendre que le DOM soit complètement chargé
document.addEventListener('DOMContentLoaded', function() {
    
    // Récupérer le formulaire
    const form = document.querySelector('form');
    
    // Récupérer les champs
    const identifiantInput = document.getElementById('identifiant');
    const motDePasseInput = document.getElementById('mot_de_passe');
    
    // Fonction pour afficher un message d'erreur
    function afficherErreur(input, message) {
        // Supprimer l'ancien message d'erreur s'il existe
        const ancienErreur = input.parentElement.querySelector('.erreur-champ');
        if (ancienErreur) {
            ancienErreur.remove();
        }
        
        // Créer le message d'erreur
        const erreur = document.createElement('div');
        erreur.className = 'erreur-champ';
        erreur.textContent = message;
        erreur.style.color = '#c33';
        erreur.style.fontSize = '14px';
        erreur.style.marginTop = '5px';
        
        // Ajouter le message après le champ
        input.parentElement.appendChild(erreur);
        
        // Mettre le champ en rouge
        input.style.borderColor = '#c33';
    }
    
    // Fonction pour effacer un message d'erreur
    function effacerErreur(input) {
        const erreur = input.parentElement.querySelector('.erreur-champ');
        if (erreur) {
            erreur.remove();
        }
        input.style.borderColor = '#e0e0e0';
    }
    
    // Validation en temps réel sur l'identifiant
    identifiantInput.addEventListener('input', function() {
        effacerErreur(this);
        
        if (this.value.length > 0 && this.value.length < 3) {
            afficherErreur(this, 'L\'identifiant doit contenir au moins 3 caractères');
        }
    });
    
    // Validation en temps réel sur le mot de passe
    motDePasseInput.addEventListener('input', function() {
        effacerErreur(this);
        
        if (this.value.length > 0 && this.value.length < 4) {
            afficherErreur(this, 'Le mot de passe doit contenir au moins 4 caractères');
        }
    });
    
    // Validation lors de la soumission du formulaire
    form.addEventListener('submit', function(event) {
        let formulaireValide = true;
        
        // Effacer toutes les erreurs précédentes
        effacerErreur(identifiantInput);
        effacerErreur(motDePasseInput);
        
        // Vérifier l'identifiant
        if (identifiantInput.value.trim() === '') {
            afficherErreur(identifiantInput, 'L\'identifiant est obligatoire');
            formulaireValide = false;
        } else if (identifiantInput.value.trim().length < 3) {
            afficherErreur(identifiantInput, 'L\'identifiant doit contenir au moins 3 caractères');
            formulaireValide = false;
        } else if (!/^[a-zA-Z0-9_]+$/.test(identifiantInput.value)) {
            afficherErreur(identifiantInput, 'L\'identifiant ne peut contenir que des lettres, chiffres et underscores');
            formulaireValide = false;
        }
        
        // Vérifier le mot de passe
        if (motDePasseInput.value === '') {
            afficherErreur(motDePasseInput, 'Le mot de passe est obligatoire');
            formulaireValide = false;
        } else if (motDePasseInput.value.length < 4) {
            afficherErreur(motDePasseInput, 'Le mot de passe doit contenir au moins 4 caractères');
            formulaireValide = false;
        }
        
        // Si le formulaire n'est pas valide, empêcher l'envoi
        if (!formulaireValide) {
            event.preventDefault();
            
            // Faire vibrer le formulaire pour indiquer l'erreur
            form.classList.add('shake');
            setTimeout(function() {
                form.classList.remove('shake');
            }, 500);
            
            // Focus sur le premier champ en erreur
            if (identifiantInput.style.borderColor === 'rgb(204, 51, 51)') {
                identifiantInput.focus();
            } else if (motDePasseInput.style.borderColor === 'rgb(204, 51, 51)') {
                motDePasseInput.focus();
            }
        }
    });
    
    // Nettoyer les espaces au début et à la fin de l'identifiant
    identifiantInput.addEventListener('blur', function() {
        this.value = this.value.trim();
    });
    
});

// Animation shake pour le formulaire
const style = document.createElement('style');
style.textContent = `
    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        10%, 30%, 50%, 70%, 90% { transform: translateX(-5px); }
        20%, 40%, 60%, 80% { transform: translateX(5px); }
    }
    
    .shake {
        animation: shake 0.5s;
    }
`;
document.head.appendChild(style);
