# 📜 Documentation : script.js - Validation du formulaire

## 🎯 Objectif

Le fichier `script.js` ajoute une **validation côté client** au formulaire de connexion pour améliorer l'expérience utilisateur en détectant les erreurs **avant** l'envoi des données au serveur.

---

## ✨ Fonctionnalités implémentées

### 1. Validation en temps réel (pendant la saisie)
- ✅ Vérification de la longueur minimale
- ✅ Messages d'erreur dynamiques
- ✅ Indication visuelle (bordure rouge)
- ✅ Disparition automatique des erreurs lors de la correction

### 2. Validation à la soumission
- ✅ Vérification des champs vides
- ✅ Vérification de la longueur minimale
- ✅ Vérification du format de l'identifiant (alphanumérique + underscore)
- ✅ Empêche l'envoi si le formulaire est invalide

### 3. Effets visuels
- ✅ Animation "shake" (tremblement) du formulaire en cas d'erreur
- ✅ Focus automatique sur le premier champ en erreur
- ✅ Bordures rouges pour les champs invalides

### 4. Nettoyage automatique
- ✅ Suppression des espaces au début et fin de l'identifiant

---

## 🔍 Règles de validation

### Identifiant
| Règle | Description |
|-------|-------------|
| **Obligatoire** | Le champ ne peut pas être vide |
| **Longueur minimale** | Au moins 3 caractères |
| **Format** | Uniquement lettres, chiffres et underscores (`a-z`, `A-Z`, `0-9`, `_`) |
| **Nettoyage** | Espaces supprimés automatiquement |

### Mot de passe
| Règle | Description |
|-------|-------------|
| **Obligatoire** | Le champ ne peut pas être vide |
| **Longueur minimale** | Au moins 4 caractères |

---

## 📋 Messages d'erreur

### Pour l'identifiant
```
- "L'identifiant est obligatoire"
- "L'identifiant doit contenir au moins 3 caractères"
- "L'identifiant ne peut contenir que des lettres, chiffres et underscores"
```

### Pour le mot de passe
```
- "Le mot de passe est obligatoire"
- "Le mot de passe doit contenir au moins 4 caractères"
```

---

## 🎬 Comportement visuel

### État normal
```css
input {
    border: 2px solid #e0e0e0;
}
```

### État d'erreur
```css
input {
    border: 2px solid #c33; /* Rouge */
}

.erreur-champ {
    color: #c33;
    font-size: 14px;
    margin-top: 5px;
}
```

### Animation shake
```css
@keyframes shake {
    0%, 100% { transform: translateX(0); }
    10%, 30%, 50%, 70%, 90% { transform: translateX(-5px); }
    20%, 40%, 60%, 80% { transform: translateX(5px); }
}
```

---

## 🔧 Fonctions principales

### `afficherErreur(input, message)`
Affiche un message d'erreur sous le champ et change la bordure en rouge.

```javascript
afficherErreur(identifiantInput, 'L\'identifiant est obligatoire');
```

### `effacerErreur(input)`
Supprime le message d'erreur et restaure la bordure normale.

```javascript
effacerErreur(identifiantInput);
```

---

## 📊 Flux de validation

### 1. Validation en temps réel (événement `input`)
```
Utilisateur tape → Vérification longueur → Affichage erreur si < minimum
```

### 2. Validation à la soumission (événement `submit`)
```
Clic sur "Se connecter"
    ↓
Effacer toutes les erreurs
    ↓
Vérifier identifiant (vide ? longueur ? format ?)
    ↓
Vérifier mot de passe (vide ? longueur ?)
    ↓
Formulaire valide ?
    ├─ OUI → Envoi au serveur
    └─ NON → Animation shake + Focus sur erreur + Blocage envoi
```

---

## 🧪 Exemples de scénarios

### ✅ Scénario valide
```
Identifiant: "admin"     → ✓ (≥3 caractères, alphanumérique)
Mot de passe: "1234"     → ✓ (≥4 caractères)
Résultat: Envoi au serveur
```

### ❌ Scénario invalide 1
```
Identifiant: "ad"        → ✗ (< 3 caractères)
Mot de passe: "1234"     → ✓
Message: "L'identifiant doit contenir au moins 3 caractères"
Résultat: Blocage + Shake
```

### ❌ Scénario invalide 2
```
Identifiant: "admin!"    → ✗ (caractère ! non autorisé)
Mot de passe: "1234"     → ✓
Message: "L'identifiant ne peut contenir que des lettres, chiffres et underscores"
Résultat: Blocage + Shake
```

### ❌ Scénario invalide 3
```
Identifiant: "admin"     → ✓
Mot de passe: "12"       → ✗ (< 4 caractères)
Message: "Le mot de passe doit contenir au moins 4 caractères"
Résultat: Blocage + Shake
```

### ❌ Scénario invalide 4
```
Identifiant: ""          → ✗ (vide)
Mot de passe: ""         → ✗ (vide)
Messages: "L'identifiant est obligatoire" + "Le mot de passe est obligatoire"
Résultat: Blocage + Shake + Focus sur identifiant
```

---

## 🛡️ Sécurité : Validation côté client vs côté serveur

### ⚠️ Important à comprendre

**La validation JavaScript (côté client) n'est PAS une sécurité !**

Elle sert uniquement à :
- ✅ Améliorer l'expérience utilisateur
- ✅ Réduire les allers-retours serveur
- ✅ Donner un feedback immédiat

### 🔐 Pourquoi la validation serveur reste indispensable

```
Un utilisateur malveillant peut :
- Désactiver JavaScript
- Modifier le code JavaScript
- Envoyer des requêtes HTTP directement (curl, Postman)
- Contourner toutes les validations côté client
```

### ✅ Architecture de sécurité en couches

```
┌─────────────────────────────────┐
│   Validation JavaScript         │  ← Confort utilisateur
│   (script.js)                   │
└─────────────────────────────────┘
              ↓
┌─────────────────────────────────┐
│   Validation PHP                │  ← VRAIE sécurité
│   (index.php - côté serveur)    │
└─────────────────────────────────┘
              ↓
┌─────────────────────────────────┐
│   Vérification hash             │  ← Authentification
│   (password_verify)             │
└─────────────────────────────────┘
```

**Les deux validations sont complémentaires, pas redondantes !**

---

## 🔄 Intégration dans le projet

### Structure des fichiers
```
/votre-projet/
├── index.php          ← Formulaire de connexion
├── script.js          ← Validation côté client (NOUVEAU)
├── domotique.php      ← Page protégée
└── generer_hash.php   ← Utilitaire
```

### Inclusion dans index.php
```html
<!-- Juste avant </body> -->
<script src="script.js"></script>
```

---

## 📱 Compatibilité navigateurs

Le script utilise des fonctionnalités modernes JavaScript :

| Fonctionnalité | Compatibilité |
|----------------|---------------|
| `addEventListener` | ✅ Tous navigateurs modernes |
| `document.querySelector` | ✅ IE9+ |
| `classList.add/remove` | ✅ IE10+ |
| Template literals | ✅ ES6 (transpiler si besoin) |
| Arrow functions | ✅ ES6 (transpiler si besoin) |

**Support :** Chrome, Firefox, Safari, Edge (toutes versions récentes)

---

## 🎨 Personnalisation

### Modifier les longueurs minimales

```javascript
// Dans script.js, ligne ~48 et ~91
if (this.value.length < 3) {  // Changer 3 en autre valeur
    afficherErreur(this, 'Au moins 3 caractères');
}
```

### Changer la couleur des erreurs

```javascript
// Dans la fonction afficherErreur, ligne ~30
erreur.style.color = '#c33';  // Remplacer par votre couleur
input.style.borderColor = '#c33';  // Idem
```

### Modifier l'animation shake

```javascript
// Dans le style dynamique, ligne ~125
@keyframes shake {
    0%, 100% { transform: translateX(0); }
    10%, 30%, 50%, 70%, 90% { transform: translateX(-10px); }  // Amplitude
    20%, 40%, 60%, 80% { transform: translateX(10px); }
}
```

### Ajouter des règles de validation

```javascript
// Exemple : mot de passe doit contenir au moins un chiffre
if (!/\d/.test(motDePasseInput.value)) {
    afficherErreur(motDePasseInput, 'Le mot de passe doit contenir au moins un chiffre');
    formulaireValide = false;
}
```

---

## 🐛 Débogage

### Tester la validation

**Ouvrez la console du navigateur (F12) et testez :**

```javascript
// Vérifier que le script est chargé
console.log('Script chargé');

// Vérifier les événements
document.querySelector('form').addEventListener('submit', function(e) {
    console.log('Formulaire soumis');
});
```

### Erreurs courantes

| Problème | Solution |
|----------|----------|
| Script ne fonctionne pas | Vérifier le chemin `<script src="script.js">` |
| Erreurs console | Ouvrir F12 et lire les messages |
| Validation ne s'applique pas | Vérifier que les IDs correspondent (`identifiant`, `mot_de_passe`) |

---

## ✅ Tests à effectuer

### Test 1 : Champs vides
1. Laisser les deux champs vides
2. Cliquer "Se connecter"
3. **Attendu :** Messages d'erreur + Animation shake

### Test 2 : Identifiant trop court
1. Saisir "ab" dans identifiant
2. Sortir du champ (blur)
3. **Attendu :** Message "au moins 3 caractères"

### Test 3 : Identifiant invalide
1. Saisir "admin!" dans identifiant
2. Cliquer "Se connecter"
3. **Attendu :** Message "lettres, chiffres et underscores"

### Test 4 : Mot de passe trop court
1. Identifiant : "admin"
2. Mot de passe : "123"
3. **Attendu :** Message "au moins 4 caractères"

### Test 5 : Validation en temps réel
1. Commencer à taper "ad"
2. **Attendu :** Message d'erreur s'affiche
3. Continuer "admin"
4. **Attendu :** Message disparaît

### Test 6 : Formulaire valide
1. Identifiant : "admin"
2. Mot de passe : "1234"
3. **Attendu :** Envoi au serveur (pas de blocage)

---

## 🎓 Pour aller plus loin

### Ajouter une confirmation de mot de passe

```javascript
const confirmInput = document.getElementById('confirm_password');

if (motDePasseInput.value !== confirmInput.value) {
    afficherErreur(confirmInput, 'Les mots de passe ne correspondent pas');
    formulaireValide = false;
}
```

### Indicateur de force du mot de passe

```javascript
function calculerForce(password) {
    let force = 0;
    if (password.length >= 8) force++;
    if (/[a-z]/.test(password)) force++;
    if (/[A-Z]/.test(password)) force++;
    if (/\d/.test(password)) force++;
    if (/[^a-zA-Z0-9]/.test(password)) force++;
    return force; // 0-5
}
```

### Captcha ou reCAPTCHA

```html
<div class="g-recaptcha" data-sitekey="YOUR_SITE_KEY"></div>
<script src="https://www.google.com/recaptcha/api.js"></script>
```

---

## 📚 Résumé

**Fichier :** `script.js`

**Rôle :** Validation côté client du formulaire de connexion

**Avantages :**
- ✅ Feedback immédiat pour l'utilisateur
- ✅ Réduction de la charge serveur
- ✅ Meilleure expérience utilisateur

**Limitations :**
- ⚠️ Ne remplace PAS la validation serveur
- ⚠️ Peut être contournée facilement

**Utilisation :**
```html
<script src="script.js"></script>
```

**Validation serveur toujours nécessaire !** 🔐
