# Système d'Authentification Sécurisé PHP

## 📋 Fichiers créés

1. **generer_hash.php** - Génère le hash du mot de passe
2. **index.php** - Page de connexion avec vérification sécurisée
3. **domotique.php** - Page protégée affichant les mesures des capteurs
4. **script.js** - Validation côté client du formulaire de connexion

## 🔧 Instructions d'installation

### Étape 1 : Générer le hash
```bash
php generer_hash.php
```

Cela affichera un hash comme :
```
$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi
```

### Étape 2 : Copier le hash dans index.php

Ouvrez `index.php` et remplacez la valeur de `$hash_stocke` par votre hash généré :

```php
$hash_stocke = 'VOTRE_HASH_ICI';
```

### Étape 3 : Placer les fichiers sur le serveur

Organisez vos fichiers comme suit :
```
/votre-site/
├── index.php
├── domotique.php
├── script.js
└── generer_hash.php
```

### Étape 4 : Tester l'application

1. Accédez à `http://localhost/` ou `http://localhost/index.php` dans votre navigateur
2. Utilisez les identifiants de test :
   - **Identifiant :** admin
   - **Mot de passe :** 1234
3. Vous serez automatiquement redirigé vers `domotique.php`

## 🏗️ Architecture de l'application

### Flux d'authentification

```
1. Utilisateur accède à index.php
   ↓
2. Saisit identifiant et mot de passe
   ↓
3. password_verify() vérifie les credentials
   ↓
4. Si OK : Création session + Redirection → domotique.php
   Si KO : Message d'erreur
   ↓
5. domotique.php vérifie la session
   ↓
6. Si session valide : Affichage des mesures
   Si session invalide : Redirection → index.php
```

### Protection des pages

**index.php** : Page publique (formulaire de connexion)
- ✅ Accessible sans authentification
- ✅ Redirige vers domotique.php si connexion réussie

**domotique.php** : Page protégée
- 🔒 Accessible uniquement si authentifié
- 🔒 Vérifie `$_SESSION['connecte']`
- 🔒 Redirige vers index.php si non authentifié

## 🔐 Fonctionnalités de sécurité

### password_hash()
- Utilise l'algorithme **bcrypt** par défaut
- Génère un **salt aléatoire** automatiquement
- Coût adaptatif (augmente avec la puissance des processeurs)

### password_verify()
- Compare le mot de passe saisi avec le hash stocké
- Résistant aux attaques par timing
- Vérifie automatiquement le salt intégré

### Protection des sessions
- Vérification de l'authentification sur chaque page protégée
- Stockage sécurisé de l'état de connexion
- Déconnexion sécurisée avec destruction de session

### Validation côté client (JavaScript)
- Vérification des champs avant envoi au serveur
- Feedback immédiat pour l'utilisateur
- Règles de validation :
  - Identifiant : minimum 3 caractères, alphanumérique + underscore
  - Mot de passe : minimum 4 caractères
- Animation visuelle en cas d'erreur
- **Note :** La validation serveur reste indispensable pour la sécurité

## 🧪 Tests à effectuer

### Test 1 : Connexion réussie et redirection
- Accéder à `index.php` (ou juste `http://localhost/`)
- Identifiant : `admin`
- Mot de passe : `1234`
- ✅ Résultat attendu : Redirection automatique vers `domotique.php`

### Test 2 : Page protégée accessible
- Après connexion, vérifier que `domotique.php` affiche :
  - Les informations utilisateur en haut à droite
  - Le tableau des mesures (température, humidité, pression, luminosité)
  - Les informations sur les capteurs
- ✅ Résultat attendu : Page complète visible

### Test 3 : Accès direct sans authentification
- Dans un nouvel onglet (navigation privée), essayer d'accéder directement à `domotique.php`
- ✅ Résultat attendu : Redirection automatique vers `index.php`

### Test 4 : Mauvais mot de passe
- Identifiant : `admin`
- Mot de passe : `incorrect`
- ❌ Résultat attendu : Message d'erreur, reste sur `index.php`

### Test 5 : Mauvais identifiant
- Identifiant : `utilisateur`
- Mot de passe : `1234`
- ❌ Résultat attendu : Message d'erreur, reste sur `index.php`

### Test 6 : Déconnexion
- Depuis `domotique.php`, cliquer sur "Se déconnecter"
- ✅ Résultat attendu : Redirection vers `index.php`
- Essayer d'accéder à nouveau à `domotique.php`
- ✅ Résultat attendu : Redirection vers `index.php` (session détruite)

## 📊 Comparaison : Avant / Après

| Aspect | Avant (MD5/SHA1) | Après (password_hash) |
|--------|------------------|----------------------|
| Algorithme | MD5 ou SHA1 | Bcrypt (via PASSWORD_DEFAULT) |
| Salt | Manuel ou absent | Automatique et aléatoire |
| Sécurité | ⚠️ Faible | ✅ Fort |
| Rainbow tables | Vulnérable | Résistant |
| Brute force | Rapide | Lent (intentionnellement) |

## 🛡️ Améliorations possibles

1. **Limitation des tentatives**
   ```php
   // Bloquer après 5 tentatives échouées
   if ($_SESSION['tentatives'] >= 5) {
       // Bloquer pendant 15 minutes
   }
   ```

2. **Protection CSRF**
   ```php
   // Générer un token CSRF
   $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
   ```

3. **HTTPS obligatoire**
   ```php
   if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
       header('Location: https://' . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
       exit;
   }
   ```

4. **Stockage en base de données**
   ```php
   // Au lieu d'un hash en dur, utiliser une BDD
   $stmt = $pdo->prepare('SELECT password_hash FROM users WHERE username = ?');
   ```

## 💡 Bonnes pratiques appliquées

- ✅ Utilisation de `password_hash()` et `password_verify()`
- ✅ Sessions PHP pour la gestion de l'authentification
- ✅ Échappement des données avec `htmlspecialchars()`
- ✅ Messages d'erreur génériques (ne révèlent pas si c'est l'identifiant ou le mot de passe qui est faux)
- ✅ Interface utilisateur intuitive et responsive

## 📝 Notes importantes

- **Ne jamais stocker les mots de passe en clair**
- **Ne jamais utiliser MD5 ou SHA1 pour les mots de passe**
- **Toujours utiliser HTTPS en production**
- **password_hash() génère un hash différent à chaque exécution** (c'est normal, le salt change)
- Le hash contient le salt, l'algorithme et le coût → tout est dans une seule chaîne

## 🔄 Changement de mot de passe

Pour changer le mot de passe :

1. Modifiez `$mot_de_passe` dans `generer_hash.php`
2. Exécutez le script pour obtenir le nouveau hash
3. Copiez le nouveau hash dans `acces.php`

Exemple pour le mot de passe "MonMotDePasseSecurise123!" :
```php
echo password_hash('MonMotDePasseSecurise123!', PASSWORD_DEFAULT);
```
